<?php

namespace Tests\Feature;

use App\Enums\ActiveEnum;
use App\Enums\VerificationEnum;
use App\Enums\VerifiedEnum;
use App\Models\Product;
use App\Models\User;
use Database\Seeders\PersonalAccessClientSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Notification;
use Tests\TestCase;

class AuthControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $seeder = PersonalAccessClientSeeder::class;

    public function test_it_can_register_a_new_user(): void
    {
        $userData = User::factory()->make([
            'password' => $password = Hash::make('password'),
            'password_confirmation' => $password,
            'verification' => VerificationEnum::EMAIL->value,
        ])->makeVisible(['password', 'password_confirmation'])
            ->toArray();

        $product = Product::factory()->create();
        $userData['product'] = $product->product_id;
        $userData['key'] = $product->key;

        $response = $this->postJson(route('auth.register'), $userData, [
            'product' => $userData['product'],
            'product-key' => $userData['key'],
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => __('User registered successfully'),
        ]);
    }

    public function test_it_sends_verification_email_when_email_verification_is_selected(): void
    {
        Notification::fake();

        $userData = User::factory()->make([
            'password' => $password = Hash::make('password'),
            'password_confirmation' => $password,
        ])->makeVisible(['password', 'password_confirmation'])
            ->toArray();
        $product = Product::factory()->create();
        $userData['product'] = $product->product_id;
        $userData['verification'] = VerificationEnum::EMAIL->value;

        $userData['product'] = $product->product_id;
        $userData['key'] = $product->key;

        $response = $this->postJson(route('auth.register'), $userData, [
            'product' => $userData['product'],
            'product-key' => $userData['key'],
        ]);

        $response->assertStatus(200);
        Notification::assertCount(1);
    }

    public function test_an_existed_user_can_login(): void
    {
        $userData = User::factory()->create([
            'active' => ActiveEnum::YES->value,
            'verified' => VerifiedEnum::YES->value,
        ]);
        $product= Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $credentials = [
            'email' => $userData->email,
            'password' => 'password',
            'product' => $product_id,
        ];

        $response = $this->postJson(route('auth.login'), $credentials, [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);
    }

    public function test_a_non_existed_user_cannot_login(): void
    {
        $userData = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $credentials = [
            'email' => $userData->email,
            'password' => 'wrong-password',
            'product' => $product_id,
        ];

        $response = $this->postJson(route('auth.login'), $credentials, [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(401);
        $response->assertJson([
            'success' => false,
        ]);
    }

    public function test_a_user_can_logout(): void
    {
        $userData = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        $this->actingAs($userData, 'api');

        $response = $this->postJson(route('auth.logout'), headers: [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);
    }

    public function test_non_verified_user_can_verify_his_account(): void
    {
        $product = Product::factory()->create();
        $user = User::factory()->create([
            'verification_code' => '123456',
            'verification' => VerificationEnum::EMAIL->value,
        ]);
        $response = $this->postJson(route('auth.verify'), [
            'verification_type' => VerificationEnum::EMAIL->value,
            'email' => $user->email,
            'code' => '123456',
        ], [
            'product' => $product->product_id,
            'product-key' => $product->key,
        ]);

        $response->assertStatus(200);
        $this->assertDatabaseHas('users', [
            'user_id' => $user->user_id,
            'verified' => VerifiedEnum::YES->value,
            'verification_code' => null,
            'verification_code_expires_at' => null,
        ]);
    }

    public function test_not_verified_user_can_request_a_new_verify_code(): void
    {
        Notification::fake();

        $product = Product::factory()->create();
        $user = User::factory()->create([
            'email' => 'shalawani84@gmail.com',
            'verification' => VerificationEnum::EMAIL->value,
            'verified' => VerifiedEnum::NO->value,
        ]);

        $response = $this->postJson(route('auth.resend-verify'), [
            'verification_type' => VerificationEnum::EMAIL->value,
            'email' => $user->email,
        ], [
            'product' => $product->product_id,
            'product-key' => $product->key,
        ]);

        $response->assertStatus(200);
        Notification::assertCount(1);
    }

    public function test_an_authenticated_user_can_refresh_the_token(): void
    {
        $user = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $this->actingAs($user, 'api');

        $response = $this->postJson(route('auth.refresh-token'), headers: [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);
    }

    public function test_an_authenticated_user_can_change_the_password(): void
    {
        $user = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $this->actingAs($user, 'api');

        $response = $this->postJson(route('auth.change-password'), [
            'old_password' => 'password',
            'password' => 'new-password',
            'password_confirmation' => 'new-password',
        ], [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);
    }
}
